//
// Copyright (c) 2020 Glauco Pacheco <glauco@cuteserver.io>
// All rights reserved
//

#ifndef VARIADIC_ARGUMENTS_PACKER_H
#define VARIADIC_ARGUMENTS_PACKER_H

#include "../TypeRegistrar.h"
#include "../QtDefinitions.h"
#include <QVariant>
#include <QVector>
#include <QList>
#include <QByteArray>
#include <QMetaType>
#include <type_traits>

namespace Cute
{

class VariadicArguments
{
public:
    template<class... ArgsTypes>
    static void packArgs(QVector<QVariant> &arguments, ArgsTypes ...Args);
    template<class... ArgsTypes>
    static void checkArgsTypes();
    template<class... ArgsTypes>
    static void fetchTypesNames(QVector<QByteArray> &parameterTypes);
};

//
// Arguments packing
//
template<class T, class... ArgsTypes>
struct ArgumentsFetcherImpl1;

template <class... ArgsTypes>
struct ArgumentsFetcherImpl2
{
    static void fetchArgs(QVector<QVariant> &arguments, int &currentArgIndex, ArgsTypes ...Args)
    {
        ArgumentsFetcherImpl1<ArgsTypes...>::fetchArgs(arguments, currentArgIndex, Args...);
    }
};

template <>
struct ArgumentsFetcherImpl2<>
{
    static void fetchArgs(QVector<QVariant> &, int &) {}
};

template<class T, class... ArgsTypes>
struct ArgumentsFetcherImpl1
{
    static void fetchArgs(QVector<QVariant> &arguments, int &currentArgIndex, T value, ArgsTypes ...Args)
    {
        arguments[currentArgIndex++] = QVariant::fromValue(value);
        ArgumentsFetcherImpl2<ArgsTypes...>::fetchArgs(arguments, currentArgIndex, Args...);
    }
};

template<class... ArgsTypes>
struct ArgumentsFetcher
{
    static void fetchArgs(QVector<QVariant> &arguments, int &currentArgIndex, ArgsTypes ...Args)
    {
        ArgumentsFetcherImpl1<ArgsTypes...>::fetchArgs(arguments, currentArgIndex, Args...);
    }
};

template<>
struct ArgumentsFetcher<>
{
    static void fetchArgs(QVector<QVariant> &arguments, int &currentArgIndex)
    {
        arguments.clear();
        currentArgIndex = 0;
    }
};

template<class... ArgsTypes>
void VariadicArguments::packArgs(QVector<QVariant> & arguments, ArgsTypes ...Args)
{
    if (sizeof...(Args) != arguments.size())
        arguments.resize(sizeof...(Args));
    int currentArgIndex = 0;
    ArgumentsFetcher<ArgsTypes...>::fetchArgs(arguments, currentArgIndex, Args...);
}

//
// Checking registration against Qt meta type system
//
template<class T>
class ArgTypeChecker
{
    typedef typename std::remove_cv<typename std::remove_reference<T>::type>::type T_BASE;
public:
    ArgTypeChecker()
    {
        Cute::registerType<T_BASE>();
        auto metaTypeId = qMetaTypeId<T_BASE>();
        Q_UNUSED(metaTypeId)
        static_assert(!std::is_pointer<T_BASE>::value, "Pointers are not allowed as argument type for remote signals/slots as well as the return type of remote slots.");
    }
};

template<>
class ArgTypeChecker<void>
{
public:
    ArgTypeChecker() = default;
};

template<class T, class... ArgsTypes>
struct ArgsTypesCheckerImpl1;

template <class... ArgsTypes>
struct ArgsTypesCheckerImpl2
{
    static void check()
    {
        ArgsTypesCheckerImpl1<ArgsTypes...>::check();
    }
};

template <>
struct ArgsTypesCheckerImpl2<>
{
    static void check() {}
};

template<class T, class... ArgsTypes>
struct ArgsTypesCheckerImpl1
{
    static void check()
    {
        ArgTypeChecker<T>();
        ArgsTypesCheckerImpl2<ArgsTypes...>::check();
    }
};

template<class... ArgsTypes>
struct ArgsTypesChecker
{
    static void check()
    {
        ArgsTypesCheckerImpl1<ArgsTypes...>::check();
    }
};

template<>
struct ArgsTypesChecker<>
{
    static void check() {}
};

template<class... ArgsTypes>
void VariadicArguments::checkArgsTypes()
{
    ArgsTypesChecker<ArgsTypes...>::check();
}

//
// Types names fetching
//
template<class T>
class TypeNameFetcher
{
    typedef typename std::remove_cv<typename std::remove_reference<T>::type>::type T_BASE;
public:
    static QByteArray typeName()
    {
        auto metaTypeId = qMetaTypeId<T_BASE>();
        Q_UNUSED(metaTypeId)
        const char *typeName = Cute::metaTypeNameFromId(qMetaTypeId<T_BASE>());
        return QByteArray::fromRawData(typeName, qstrlen(typeName));
    }
};

template<>
class TypeNameFetcher<void>
{
public:
    static QByteArray typeName()
    {
        const char *typeName = "void";
        return QByteArray::fromRawData(typeName, qstrlen(typeName));
    }

};

template<class T, class... ArgsTypes>
struct TypesNamesFetcherImpl1;

template <class... ArgsTypes>
struct TypesNamesFetcherImpl2
{
    static void fetchNames(QVector<QByteArray> &parameterTypes, int &currentArgIndex)
    {
        TypesNamesFetcherImpl1<ArgsTypes...>::fetchNames(parameterTypes, currentArgIndex);
    }
};

template <>
struct TypesNamesFetcherImpl2<>
{
    static void fetchNames(QVector<QByteArray> &, int &) {}
};

template<class T, class... ArgsTypes>
struct TypesNamesFetcherImpl1
{
    static void fetchNames(QVector<QByteArray> &parameterTypes, int &currentArgIndex)
    {
        parameterTypes[currentArgIndex++] = TypeNameFetcher<T>::typeName();
        TypesNamesFetcherImpl2<ArgsTypes...>::fetchNames(parameterTypes, currentArgIndex);
    }
};

template<class... ArgsTypes>
struct TypesNamesFetcher
{
    static void fetchNames(QVector<QByteArray> &parameterTypes, int &currentArgIndex)
    {
        TypesNamesFetcherImpl1<ArgsTypes...>::fetchNames(parameterTypes, currentArgIndex);
    }
};

template<>
struct TypesNamesFetcher<>
{
    static void fetchNames(QVector<QByteArray> &parameterTypes, int &currentArgIndex)
    {
        parameterTypes.clear();
        currentArgIndex = 0;
    }
};

template<class... ArgsTypes>
void VariadicArguments::fetchTypesNames(QVector<QByteArray> &parameterTypes)
{
    if (sizeof...(ArgsTypes) != parameterTypes.size())
    parameterTypes.resize(sizeof...(ArgsTypes));
    int currentArgIndex = 0;
    TypesNamesFetcher<ArgsTypes...>::fetchNames(parameterTypes, currentArgIndex);
}

}

#endif // VARIADIC_ARGUMENTS_PACKER_H
