//
// Copyright (c) 2020 Glauco Pacheco <glauco@cuteserver.io>
// All rights reserved
//

#ifndef CUTE_SERVER_SDK_INTERNAL_H
#define CUTE_SERVER_SDK_INTERNAL_H

#include "../IConnectionInformation.h"
#include "HttpBroker.h"
#include "HttpRequest.h"
#include "HttpResponse.h"
#include <QObject>
#include <QString>
#include <QSharedPointer>

namespace Cute::Server
{
    class IRegisteredRemoteObject;
}

PUBLIC void registerRemoteObject(const ::Cute::Server::IRegisteredRemoteObject *remoteObject) noexcept;

namespace Cute::Server
{

class IRegisteredRemoteObject
{
public:
    IRegisteredRemoteObject() = default;
    virtual ~IRegisteredRemoteObject() = default;
    [[nodiscard]] virtual const QMetaObject* staticMetaObject() const = 0;
    [[nodiscard]] virtual const QString &resourcePath() const = 0;
    [[nodiscard]] virtual const QString &hostnames() const = 0;
    [[nodiscard]] virtual QObject* createInstance(IConnectionInformation*) const = 0;
    [[nodiscard]] virtual QObject* createInstance(QSharedPointer<IConnectionInformation>) const = 0;
};

template <class T_RemoteObject>
class RegisteredRemoteObject : public IRegisteredRemoteObject
{
public:
    RegisteredRemoteObject(const char *resourcePath, const char *hostnames) noexcept : m_resourcePath(QString::fromLatin1(resourcePath).toLower()), m_hostnames(QString::fromLatin1(hostnames))
    {
        registerRemoteObject(this);
    }
    ~RegisteredRemoteObject() override = default;
    [[nodiscard]] const QMetaObject* staticMetaObject() const override {return &T_RemoteObject::staticMetaObject;}
    [[nodiscard]] const QString &resourcePath() const override {return m_resourcePath;}
    [[nodiscard]] const QString &hostnames() const override {return m_hostnames;}
    [[nodiscard]] QObject* createInstance(IConnectionInformation *connectionInformation) const override {return new T_RemoteObject(QSharedPointer<IConnectionInformation>(connectionInformation));}
    [[nodiscard]] QObject* createInstance(QSharedPointer<IConnectionInformation> connectionInformation) const override {return new T_RemoteObject(connectionInformation);}

private:
    const QString m_resourcePath;
    const QString m_hostnames;
};

}

#define MERGE_PRIVATE(a,b) a##b
#define MERGE(a,b) MERGE_PRIVATE(a,b)
#define UNIQUE_IDENTIFIER MERGE(registeredRemoteObject, __LINE__)

#define REGISTER_REMOTE_OBJECT_WITHOUT_HOST_ADDRESSES(AbsoluteResourcePath, RemoteObjectTypeName) \
    static ::Cute::Server::RegisteredRemoteObject<RemoteObjectTypeName> UNIQUE_IDENTIFIER(AbsoluteResourcePath, "*");

#define REGISTER_REMOTE_OBJECT_WITH_HOST_ADDRESSES(AbsoluteResourcePath, HostAddresses, RemoteObjectTypeName) \
    static ::Cute::Server::RegisteredRemoteObject<RemoteObjectTypeName> UNIQUE_IDENTIFIER(AbsoluteResourcePath, HostAddresses);

#define FETCH_REGISTER_REMOTE_OBJECT_MACRO_FROM(val1, val2, val_or_macro, macro, ...) macro
#define FETCH_REGISTER_REMOTE_OBJECT_MACRO(...) \
    FETCH_REGISTER_REMOTE_OBJECT_MACRO_FROM(__VA_ARGS__, REGISTER_REMOTE_OBJECT_WITH_HOST_ADDRESSES, REGISTER_REMOTE_OBJECT_WITHOUT_HOST_ADDRESSES)

#define REGISTER_REMOTE_OBJECT(...) FETCH_REGISTER_REMOTE_OBJECT_MACRO(__VA_ARGS__)(__VA_ARGS__)

#ifndef Q_MOC_RUN
#define REMOTE_SIGNAL
#define REMOTE_SLOT
#define HTTP_DELETE
#define HTTP_GET
#define HTTP_PATCH
#define HTTP_POST
#define HTTP_PUT
#endif

#endif // CUTE_SERVER_SDK_INTERNAL_H
