//
// Copyright (c) 2021 Glauco Pacheco <glauco@cuteserver.io>
// All rights reserved
//

#ifndef CUTE_QT_DEFINITIONS_H
#define CUTE_QT_DEFINITIONS_H

#include <QTextStream>
#include <QAbstractSocket>
#include <QList>
#include <QSslError>
#include <QSslSocket>
#include <QPair>
#include <QMetaType>
#include <QFile>
#if defined(__unix__) || defined(__APPLE__)
#include <unistd.h>
#include <cerrno>
#elif WIN32
#include "SDK.h"
#endif


namespace Cute
{

#if QT_VERSION < QT_VERSION_CHECK(6, 0, 0)
typedef int SizeType;
#else
typedef qsizetype SizeType;
#endif

#if QT_VERSION < QT_VERSION_CHECK(5, 15, 0)
#define Q_TCP_SOCKET_ERROR_SIGNAL static_cast<void (QAbstractSocket::*)(QAbstractSocket::SocketError)>(&QTcpSocket::error)
#else
#define Q_TCP_SOCKET_ERROR_SIGNAL &QTcpSocket::errorOccurred
#endif

#define Q_WEB_SOCKET_ERROR_SIGNAL static_cast<void (QWebSocket::*)(QAbstractSocket::SocketError)>(&QWebSocket::error)

inline QTextStream &endl(QTextStream &in)
{
#if QT_VERSION < QT_VERSION_CHECK(5, 14, 0)
    return ::endl(in);
#else
    return Qt::endl(in);
#endif
}

inline static QList<QSslError> sslErrorsFrom(const QSslSocket &sslSocket)
{
#if QT_VERSION < QT_VERSION_CHECK(5, 15, 0)
    return sslSocket.sslErrors();
#else
    return sslSocket.sslHandshakeErrors();
#endif
}

template <typename T>
void registerMetaTypeStreamOperators()
{
#if QT_VERSION < QT_VERSION_CHECK(6, 0, 0)
    qRegisterMetaTypeStreamOperators<T>();
#endif
}

template <typename T>
void registerMetaTypeStreamOperators(const char *typeName)
{
#if QT_VERSION < QT_VERSION_CHECK(6, 0, 0)
    qRegisterMetaTypeStreamOperators<T>(typeName);
#endif
}

template <typename T1, typename T2>
QPair<T1, T2> makePair(const T1 &value1, const T2 &value2)
{
#if QT_VERSION < QT_VERSION_CHECK(6, 0, 0)
    return qMakePair(value1, value2);
#else
    return std::make_pair(value1, value2);
#endif
}

inline int metaTypeIdFromName(const QByteArray &typeName)
{
#if QT_VERSION < QT_VERSION_CHECK(6, 0, 0)
    return QMetaType::type(typeName);
#else
    return QMetaType::fromName(typeName).id();
#endif
}

inline int metaTypeIdFromName(const char *typeName)
{
#if QT_VERSION < QT_VERSION_CHECK(6, 0, 0)
    return QMetaType::type(typeName);
#else
    return QMetaType::fromName(typeName).id();
#endif
}

inline QByteArray metaTypeNameFromId(const int typeId)
{
#if QT_VERSION < QT_VERSION_CHECK(6, 0, 0)
    return QMetaType::typeName(typeId);
#else
    return QMetaType(typeId).name();
#endif
}

inline QByteArray metaTypeNameFromName(const char *typeName)
{
    return metaTypeNameFromId(metaTypeIdFromName(typeName));
}

template <typename T>
QByteArray metaTypeNameFromType()
{
    return Cute::metaTypeNameFromId(qMetaTypeId<T>());
}

class UniquePtrQObjectDeleteLater
{
public:
    void operator()(QObject *object) {if (object) {object->deleteLater();}}
};

// As per QTBUG-8544, QFile::flush only flushes Qt-managed
// buffers to C library managed buffers. Thus, we need a
// function to call the native flush for UNIX and Windows
#if defined(__unix__) || defined(__APPLE__)
inline void NativeFileFlush(QFile &file)
{
    file.flush();
    int result;
    int counter = 0;
    do
    {
        result = fsync(file.handle());
    }
    while (-1 == result && EIO == errno && ++counter < 10);
}
#elif defined(_WIN32) || defined(_WIN64)
PUBLIC void NativeFileFlush(QFile &file);
#endif
}

#endif // CUTE_QT_DEFINITIONS_H
