//
// Copyright (c) 2020 Glauco Pacheco <glauco@cuteserver.io>
// All rights reserved
//

#ifndef CUSTOM_TYPE_REGISTRAR_H
#define CUSTOM_TYPE_REGISTRAR_H

#include "DataStream/DataStream.h"
#include <QMetaType>
#include <QReadWriteLock>
#include <QHash>
#include <QSharedPointer>
#include <QMutex>
#include <QMutexLocker>
#include <type_traits>

namespace Cute
{

class PUBLIC ICuteStreamHelper
{
public:
    ICuteStreamHelper() = default;
    ~ICuteStreamHelper() = default;
    virtual void load(DataStream &s, void *data) const = 0;
    virtual void save(DataStream &s, void *data) const = 0;
    static QReadWriteLock &mainReadWriteLock();
    static QHash<int, QSharedPointer<ICuteStreamHelper>> &registeredStreamHelpers();
};

template <typename T>
class CuteStreamHelper : public ICuteStreamHelper
{
public:
    CuteStreamHelper() = default;
    ~CuteStreamHelper() = default;
    void load(DataStream &s, void *data) const override {s >> *static_cast<T*>(data);}
    void save(DataStream &s, void *data) const override {s << *static_cast<T*>(data);}
};

void PUBLIC registerStandardTypes();

template <typename T>
int registerType()
{
    registerStandardTypes();
    typedef typename std::remove_cv<typename std::remove_reference<T>::type>::type T_BASE;
    const auto id = qRegisterMetaType<T_BASE>();
    static_assert(!std::is_pointer<T_BASE>::value, "Pointers are not allowed as argument type for remote signals/slots as well as the return type of remote slots.");
    QWriteLocker locker(&ICuteStreamHelper::mainReadWriteLock());
    if (!ICuteStreamHelper::registeredStreamHelpers().contains(id))
        ICuteStreamHelper::registeredStreamHelpers().insert(id, QSharedPointer<ICuteStreamHelper>(new CuteStreamHelper<T_BASE>));
    return id;
}

bool PUBLIC isArgumentTypeValid(const QByteArray &typeName);
QByteArray PUBLIC integralTypename(const QByteArray &typeName);
bool PUBLIC isTypeRegistered(const QByteArray &typeName);

template <typename T>
bool isTypeRegistered()
{
    typedef typename std::remove_cv<typename std::remove_reference<T>::type>::type T_BASE;
    static_assert(!std::is_pointer<T_BASE>::value, "Pointers are not allowed as argument type for remote signals/slots as well as the return type of remote slots.");
    const auto id = qMetaTypeId<T_BASE>();
    if (id <= 0)
        return false;
    QWriteLocker locker(&ICuteStreamHelper::mainReadWriteLock());
    return ICuteStreamHelper::registeredStreamHelpers().contains(id);
}

#define STATIC_META_TYPE_ID_FCN(Type) \
static int staticMetaTypeId() \
{ \
    static bool firstRun = true; \
    static int id = 0; \
    static QMutex lock; \
    QMutexLocker locker(&lock); \
    if (firstRun) \
    { \
        firstRun = false; \
        id = Cute::registerType<Type>(); \
    } \
    return id; \
}

}

Q_DECLARE_METATYPE(std::string)
Q_DECLARE_METATYPE(std::u16string)
Q_DECLARE_METATYPE(std::u32string)
Q_DECLARE_METATYPE(char16_t)
Q_DECLARE_METATYPE(char32_t)
Q_DECLARE_SEQUENTIAL_CONTAINER_METATYPE(std::deque)
Q_DECLARE_SEQUENTIAL_CONTAINER_METATYPE(std::unordered_set)
Q_DECLARE_ASSOCIATIVE_CONTAINER_METATYPE(std::unordered_map)
Q_DECLARE_ASSOCIATIVE_CONTAINER_METATYPE(std::multimap)
Q_DECLARE_ASSOCIATIVE_CONTAINER_METATYPE(std::unordered_multimap)
Q_DECLARE_ASSOCIATIVE_CONTAINER_METATYPE(QMultiMap)
Q_DECLARE_ASSOCIATIVE_CONTAINER_METATYPE(QMultiHash)

#endif // CUSTOM_TYPE_REGISTRAR_H
