//
// Copyright (c) 2021 Glauco Pacheco <glauco@cuteserver.io>
// All rights reserved
//

//
// DataStream serves two purposes. First, it has the main responsibility
// of providing "safe" data streaming. As it is applied to data coming from
// the wire, messages can be crafted by malicious users to abuse memory
// allocation. QDataStream optimizes data streaming by reserving memory
// prior to data processing. This mechanism is based on trust - something
// that do not exist when processing network data. This reservation uses
// the value of a 4-byte unsigned integer without any checks/restrictions.
// Thus, for example, a malicious user can send a message pretending to be a streamed
// QVector, but with a size of 4,294,967,295 bytes. Note that the message itself can be
// small. Just the size of the message must have the maximum allowed value for an
// unsigned 4-byte integer to force an excessive memory allocation. Another
// allocations limit memory growth to 1024**2 bytes (1MB) per step, which can also be
// used to abuse memory allocation.
//
// DataStream supports most types natively supported by QVariant. The exception comes from
// types belonging to GUI as QEasingCurve, from types that can be represented by other
// types (for example, instead of sending a QJsonDocument, a QByteArray of the UTF-8 representation
// can be sent instead) and types that ceased to exist in Qt6 like QRegExp, QLinkedList and QPair.
// DataStream also supports C++ standard types (std::pair, std::vector, std::list, std::deque, std::unordered_set,
// std::queue, std::stack, std::map, std::multimap, std::unordered_map, std::unordered_multimap,
// std::string, std::u16string and std::u32string). DataStream abstracts away implementation details
// regarding Qt containers allowing them to be exchanged by peers running different major
// Qt versions (5/6).
// DataStream supports custom types. Like in Qt, stream operators must be defined with the following
// signatures:
//     DataStream &operator<<(DataStream &out, const MyClass &myObj);
//     DataStream &operator>>(DataStream &in, MyClass &myObj);
//
// Custom types must be registered using Cute::registerType. This function allows the custom
// type to be represented as a QVariant and a QVariant holding the custom type to be streamed
// by DataStream. Supported template-based types from Qt and C++ standard library (QVector, QList,
// std::list, ...) must also be registered. However, as template-based stream operators are
// already defined for these types, they are generated automatically upon registration and thus don't
// have to be provided.
//
// In Qt5, std::queue and std::stack must be declared using Q_DECLARE_METATYPE with the instantiated
// class, like Q_DECLARE_METATYPE(std::queue<uint>). Qt6 does not require std::queue and std::stack
// to be declared. Note that when declaring std::queue or std::stack when using Qt5, the template type
// must have the name used by Qt meta type system. For example, quint16 has typename ushort. Thus,
// in that case, std::stack should be declared as Q_DECLARE_METATYPE(std::stack<ushort>) instead of
// Q_DECLARE_METATYPE(std::stack<quint16>).
//

#ifndef CUTE_DATA_STREAM_H
#define CUTE_DATA_STREAM_H

#include "../SDK.h"
#include "../QtDefinitions.h"
#include <QDataStream>
#include <QIODevice>
#include <QVariant>
#include <QFloat16>
#include <QByteArray>
#include <QString>
#include <QBitArray>
#include <QList>
#include <QVector>
#include <QSet>
#include <QHash>
#include <QMultiHash>
#include <QMap>
#include <QMultiMap>
#include <QQueue>
#include <QStack>
#include <QDate>
#include <QTime>
#include <QDateTime>
#include <QTimeZone>
#include <QLocale>
#include <QUrl>
#include <QRegularExpression>
#include <QUuid>
#include <type_traits>
#include <cmath>
#include <utility>
#include <list>
#include <deque>
#include <string>
#include <vector>
#include <unordered_set>
#include <stack>
#include <queue>
#include <map>
#include <unordered_map>


namespace Cute
{

class PUBLIC DataStream
{
public:
    enum class Version : quint32 {Cute_1_0 = 1, Cute_DefaultCompiledVersion = Cute_1_0};
    explicit DataStream(QIODevice *ioDevice);
    ~DataStream() = default;
    [[nodiscard]] Version version() const {return m_version;}
    void setVersion(Version version) {m_version = version;}
    void setValid(bool isValid) {m_isOk = isValid;}
    qint64 writeData(const char *data, qint64 length);
    qint64 readData(char *data, qint64 length);
    void resetStatus();
    [[nodiscard]] qint32 containerReserveLimit() const {return m_containerReserveLimit;}
    void setContainerReserveLimit(qint32 containerReserveLimit);
    [[nodiscard]] QIODevice *device() const {return m_ioDevice;}
    [[nodiscard]] bool isValid() const {return m_ioDevice && m_isOk;}
    [[nodiscard]] bool atEnd() const {return m_ioDevice && m_ioDevice->atEnd();}
    //
    // Basic C++ types
    //
    DataStream &operator<<(qint8 i);
    DataStream &operator>>(qint8 &i);
    DataStream &operator<<(quint8 i);
    DataStream &operator>>(quint8 &i);
    DataStream &operator<<(qint16 i);
    DataStream &operator>>(qint16 &i);
    DataStream &operator<<(quint16 i);
    DataStream &operator>>(quint16 &i);
    DataStream &operator<<(qint32 i);
    DataStream &operator>>(qint32 &i);
    DataStream &operator<<(quint32 i);
    DataStream &operator>>(quint32 &i);
    DataStream &operator<<(qint64 i);
    DataStream &operator>>(qint64 &i);
    DataStream &operator<<(quint64 i);
    DataStream &operator>>(quint64 &i);
    DataStream &operator<<(float f);
    DataStream &operator>>(float &f);
    DataStream &operator<<(double d);
    DataStream &operator>>(double &d);
    DataStream &operator<<(qfloat16 f);
    DataStream &operator>>(qfloat16 &f);
    DataStream &operator<<(std::nullptr_t) {return *this;}
    DataStream &operator>>(std::nullptr_t &ptr) {ptr = nullptr; return *this;}
    DataStream &operator<<(bool b);
    DataStream &operator>>(bool &b);
    DataStream &operator<<(char16_t c);
    DataStream &operator>>(char16_t &c);
    DataStream &operator<<(char32_t c);
    DataStream &operator>>(char32_t &c);
    DataStream &operator<<(char c);
    DataStream &operator>>(char &c);
    //
    // Basic Qt types
    //
    DataStream &operator<<(QChar c);
    DataStream &operator>>(QChar &c);
    DataStream &operator<<(const QBitArray &ba);
    DataStream &operator>>(QBitArray &ba);
    DataStream &operator<<(const QUrl &url);
    DataStream &operator>>(QUrl &url);
    DataStream &operator<<(const QUuid &id);
    DataStream &operator>>(QUuid &id);
    DataStream &operator<<(const QDate &date);
    DataStream &operator>>(QDate &date);
    DataStream &operator<<(const QTime &time);
    DataStream &operator>>(QTime &time);
    DataStream &operator<<(const QTimeZone &tz);
    DataStream &operator>>(QTimeZone &tz);
    DataStream &operator<<(const QDateTime &dt);
    DataStream &operator>>(QDateTime &dt);
    DataStream &operator<<(const QLocale &l);
    DataStream &operator>>(QLocale &l);
    DataStream &operator<<(const QRegularExpression &re);
    DataStream &operator>>(QRegularExpression &re);
    template <typename Enum>
    DataStream &operator<<(QFlags<Enum> e);
    template <typename Enum>
    DataStream &operator>>(QFlags<Enum> &e);
    DataStream &operator<<(const QVariant &variant);
    DataStream &operator>>(QVariant &variant);
    //
    // UTF-8 string containers
    //
    DataStream &operator<<(const QByteArray &ba);
    DataStream &operator>>(QByteArray &ba);
    DataStream &operator<<(const std::string &str);
    DataStream &operator>>(std::string &str);
    //
    // UTF-16 string containers
    //
    DataStream &operator<<(const QString &str);
    DataStream &operator>>(QString &str);
    DataStream &operator<<(const std::u16string &str);
    DataStream &operator>>(std::u16string &str);
    //
    // UTF-32 string containers
    //
    DataStream &operator<<(const std::u32string &str);
    DataStream &operator>>(std::u32string &str);
    //
    // Sequence containers
    //
#if QT_VERSION < QT_VERSION_CHECK(6,0,0)
    DataStream &operator<<(const QStringList &strList);
    DataStream &operator>>(QStringList &strList);
    template <class T>
    DataStream &operator<<(const QVector<T> &c);
    template <class T>
    DataStream &operator>>(QVector<T> &c);
#endif
    template <typename T>
    DataStream &operator<<(const QList<T> &list);
    template <typename T>
    DataStream &operator>>(QList<T> &list);
    template <typename T>
    DataStream &operator<<(const std::vector<T> &v);
    template <typename T>
    DataStream &operator>>(std::vector<T> &v);
    template <typename T>
    DataStream &operator<<(const std::list<T> &l);
    template <typename T>
    DataStream &operator>>(std::list<T> &l);
    template <typename T>
    DataStream &operator<<(const std::deque<T> &d);
    template <typename T>
    DataStream &operator>>(std::deque<T> &d);
    //
    // Unordered Set Containers
    //
    template <typename T>
    DataStream &operator<<(const QSet<T> &set);
    template <typename T>
    DataStream &operator>>(QSet<T> &set);
    template <typename T>
    DataStream &operator<<(const std::unordered_set<T> &set);
    template <typename T>
    DataStream &operator>>(std::unordered_set<T> &set);
    //
    // Stack containers
    //
    template <class T>
    DataStream &operator<<(const QStack<T> &stack);
    template <class T>
    DataStream &operator>>(QStack<T> &stack);
    template <class T>
    DataStream &operator<<(const std::stack<T> &stack);
    template <class T>
    DataStream &operator>>(std::stack<T> &stack);
    //
    // Queue containers
    //
    template <typename T>
    DataStream &operator<<(const QQueue<T> &q);
    template <typename T>
    DataStream &operator>>(QQueue<T> &q);
    template <typename T>
    DataStream &operator<<(const std::queue<T> &q);
    template <typename T>
    DataStream &operator>>(std::queue<T> &q);
    //
    // Associative containers
    //
    template <class Key, class T>
    DataStream &operator<<(const QMap<Key, T> &map);
    template <class Key, class T>
    DataStream &operator>>(QMap<Key, T> &map);
    template <class Key, class T>
    DataStream &operator<<(const QHash<Key, T> &hash);
    template <class Key, class T>
    DataStream &operator>>(QHash<Key, T> &hash);
    template <class Key, class T>
    DataStream &operator<<(const std::map<Key, T> &map);
    template <class Key, class T>
    DataStream &operator>>(std::map<Key, T> &map);
    template <class Key, class T>
    DataStream &operator<<(const std::unordered_map<Key, T> &map);
    template <class Key, class T>
    DataStream &operator>>(std::unordered_map<Key, T> &map);
    template <class Key, class T>
    DataStream &operator<<(const QMultiMap<Key, T> &map);
    template <class Key, class T>
    DataStream &operator>>(QMultiMap<Key, T> &map);
    template <class Key, class T>
    DataStream &operator<<(const QMultiHash<Key, T> &hash);
    template <class Key, class T>
    DataStream &operator>>(QMultiHash<Key, T> &hash);
    template <class Key, class T>
    DataStream &operator<<(const std::multimap<Key, T> &map);
    template <class Key, class T>
    DataStream &operator>>(std::multimap<Key, T> &map);
    template <class Key, class T>
    DataStream &operator<<(const std::unordered_multimap<Key, T> &map);
    template <class Key, class T>
    DataStream &operator>>(std::unordered_multimap<Key, T> &map);
    //
    // std::pair
    //
    template <typename T1, typename T2>
    DataStream &operator<<(const std::pair<T1, T2> &p);
    template <typename T1, typename T2>
    DataStream &operator>>(std::pair<T1, T2> &p);

private:
    template<class T>
    qint64 writeNumber(const T &obj);
    template<class T>
    qint64 readNumber(T &obj);

private:
    Version m_version = Version::Cute_1_0;
    qint32 m_containerReserveLimit = 4096;
    QIODevice *m_ioDevice = nullptr;
    const bool m_isBigEndian = false;
    bool m_isOk = true;
    const QList<QByteArray> m_availableTimeZoneIdsData = QTimeZone::availableTimeZoneIds();
    const QDateTime m_dateTime = QDateTime::currentDateTimeUtc();
};

template<class T>
qint64 DataStream::writeNumber(const T &obj)
{
    static_assert(std::is_arithmetic<T>::value, "Guess what! DataStream::writeNumber only accept numbers.");
    if (!isValid())
        return -1;
    else if (m_isBigEndian || 1 == sizeof(obj))
        return writeData(reinterpret_cast<const char*>(&obj), sizeof(obj));
    else
    {
        T swappedObj;
        const auto objPtr = reinterpret_cast<const quint8*>(&obj);
        auto swappedObjPtr = reinterpret_cast<quint8*>(&swappedObj);
        for (int i = 0; i < sizeof(T); ++i)
            swappedObjPtr[i] = objPtr[sizeof(T) - 1 - i];
        return writeData(reinterpret_cast<const char*>(&swappedObj), sizeof(swappedObj));
    }
}

template<class T>
qint64 DataStream::readNumber(T &obj)
{
    static_assert(std::is_arithmetic<T>::value, "Guess what! DataStream::readNumber only accept numbers.");
    obj = T(0);
    if (!isValid())
        return -1;
    T readObj;
    const auto readBytes = readData(reinterpret_cast<char*>(&readObj), sizeof(readObj));
    if (readBytes != sizeof(readObj))
        return -1;
    if (m_isBigEndian || 1 == sizeof(obj))
        obj = readObj;
    else
    {
        T swappedObj;
        const auto readObjPtr = reinterpret_cast<const quint8*>(&readObj);
        auto swappedObjPtr = reinterpret_cast<quint8*>(&swappedObj);
        for (int i = 0; i < sizeof(T); ++i)
            swappedObjPtr[i] = readObjPtr[sizeof(T) - 1 - i];
        obj = swappedObj;
    }
    if (!std::isfinite(static_cast<double>(obj)) || (std::is_integral<T>::value && T(obj) != obj))
    {
        obj = T(0);
        return -1;
    }
    if (obj != T(0) && ((obj+T(1)) == T(1)))
        obj = T(0);
    return readBytes;
}

// Helper methods
namespace HelperMethods
{
inline bool canReadBytesFromStream(const DataStream &stream, quint64 bytesCount)
{
    return stream.isValid() && bytesCount <= static_cast<quint64>(stream.device()->bytesAvailable());
}

template <typename T>
bool readFromStream(DataStream &stream, T &value)
{
    if (!stream.isValid())
        return false;
    stream >> value;
    return stream.isValid();
}

}

// UTF-8 string containers (QByteArray, std::string)
template <typename T>
class Utf8StringContainer
{
    static_assert(sizeof(T) == 0, "Utf8StringContainer only supports specializations for the following types: QByteArray, std::string.");
};

template <>
class Utf8StringContainer<QByteArray>
{
public:
    Utf8StringContainer() = delete;
    ~Utf8StringContainer() = delete;
    static bool save(DataStream &stream, const QByteArray &ba)
    {
        quint64 length = ba.isNull() ? UINT64_MAX : ba.size();
        stream << length;
        if (0 < length && length < UINT64_MAX)
            stream.writeData(ba.data(), ba.size());
        return stream.isValid();
    }

    static bool load(DataStream &stream, QByteArray &ba)
    {
        ba.resize(0);
        if (!stream.isValid())
            return false;
        quint64 length;
        if (!HelperMethods::readFromStream(stream, length))
            return false;
        switch (length)
        {
            case 0:
                ba.resize(0);
                return true;
            case UINT64_MAX:
                ba = QByteArray(nullptr);
                return true;
            default:
                if (HelperMethods::canReadBytesFromStream(stream, length))
                {
                    ba.resize(static_cast<Cute::SizeType>(length));
                    if (length == stream.readData(ba.data(), static_cast<qint64>(length)))
                        return true;
                }
                stream.setValid(false);
                ba.resize(0);
                return false;
        }
    }
};

template <>
class Utf8StringContainer<std::string>
{
public:
    Utf8StringContainer() = delete;
    ~Utf8StringContainer() = delete;
    static bool save(DataStream &stream, const std::string &str)
    {
        quint64 length = str.empty() ? UINT64_MAX : str.length();
        stream << length;
        if (0 < length && length < UINT64_MAX)
            stream.writeData(str.data(), static_cast<qint64>(str.size()));
        return stream.isValid();
    }

    static bool load(DataStream &stream, std::string &str)
    {
        str.clear();
        if (!stream.isValid())
            return false;
        quint64 length;
        if (!HelperMethods::readFromStream(stream, length))
            return false;
        switch (length)
        {
            case 0:
            case UINT64_MAX:
                str.clear();
                return true;
            default:
                if (HelperMethods::canReadBytesFromStream(stream, length))
                {
                    str.resize(length);
                    if (length == stream.readData(str.data(), static_cast<qint64>(length)))
                        return true;
                }
                stream.setValid(false);
                str.clear();
                return false;
        }
    }
};

// UTF-16 string containers (QString, std::u16string)
template <typename T>
class Utf16StringContainer
{
    static_assert(sizeof(T) == 0, "Utf16StringContainer only supports specializations for the following types: QString, std::u16string.");
};

template <>
class Utf16StringContainer<QString>
{
public:
    Utf16StringContainer() = delete;
    ~Utf16StringContainer() = delete;
    static bool save(DataStream &stream, const QString &str)
    {
        quint64 length = str.isNull() ? UINT64_MAX : str.size();
        stream << length;
        if (0 < length && length < UINT64_MAX)
        {
            for (const QChar &ch: str)
                stream << static_cast<quint16>(ch.unicode());
        }
        return stream.isValid();
    }

    static bool load(DataStream &stream, QString &str)
    {
        str.resize(0);
        if (!stream.isValid())
            return false;
        quint64 length;
        if (!HelperMethods::readFromStream(stream, length))
            return false;
        switch (length)
        {
            case 0:
                str.resize(0);
                return true;
            case UINT64_MAX:
                str = QString();
                return true;
            default:
                if (HelperMethods::canReadBytesFromStream(stream, sizeof(quint16) * length))
                {
                    str.reserve(static_cast<Cute::SizeType>(length));
                    for (int i = 0; i < length; ++i)
                    {
                        quint16 unicode;
                        stream >> unicode;
                        str.append(QChar(unicode));
                        if (!stream.isValid())
                        {
                            str.resize(0);
                            break;
                        }
                    }
                }
                else
                    stream.setValid(false);
                return stream.isValid();
        }
    }
};

template <>
class Utf16StringContainer<std::u16string>
{
public:
    Utf16StringContainer() = delete;
    ~Utf16StringContainer() = delete;
    static bool save(DataStream &stream, const std::u16string &str)
    {
        quint64 length = str.empty() ? UINT64_MAX : str.size();
        stream << length;
        if (0 < length && length < UINT64_MAX)
        {
            for (const char16_t &ch : str)
                stream << static_cast<quint16>(ch);
        }
        return stream.isValid();
    }

    static bool load(DataStream &stream, std::u16string &str)
    {
        str.clear();
        if (!stream.isValid())
            return false;
        quint64 length;
        if (!HelperMethods::readFromStream(stream, length))
            return false;
        switch (length)
        {
            case 0:
                str.clear();
                return true;
            case UINT64_MAX:
                str = std::u16string();
                return true;
            default:
                if (HelperMethods::canReadBytesFromStream(stream, sizeof(quint16) * length))
                {
                    str.reserve(length);
                    for (int i = 0; i < length; ++i)
                    {
                        quint16 unicode;
                        stream >> unicode;
                        str.push_back(static_cast<char16_t>(unicode));
                        if (!stream.isValid())
                        {
                            str.resize(0);
                            break;
                        }
                    }
                }
                else
                    stream.setValid(false);
                return stream.isValid();
        }
    }
};

// Sequence containers (QVector, QStringList, QList, std::vector, std::list, std::deque)
template <typename T>
class SequenceContainer
{
    static_assert(sizeof(T) == 0, "SequenceContainer only supports specializations for the following types: QVector, QStringList, QList, std::vector, std::list and std::deque.");
};

#if QT_VERSION < QT_VERSION_CHECK(6,0,0)
template <typename T>
class SequenceContainer<QVector<T>>
{
public:
    SequenceContainer() = delete;
    ~SequenceContainer() = delete;
    static bool save(DataStream &stream, const QVector<T> &c)
    {
        if (!stream.isValid())
            return false;
        stream << static_cast<quint64>(c.size());
        QVectorIterator<T> i(c);
        while (i.hasNext())
            stream << i.next();
        return stream.isValid();
    }

    static bool load(DataStream &stream, QVector<T> &c)
    {
        c.resize(0);
        if (!stream.isValid())
            return false;
        quint64 size;
        if (!HelperMethods::readFromStream(stream, size))
            return false;
        if (0 < size)
        {
            if (!HelperMethods::canReadBytesFromStream(stream, size))
            {
                stream.setValid(false);
                return false;
            }
            c.reserve(qMin<Cute::SizeType>(stream.containerReserveLimit(), size));
            for (quint64 i = 0; i < size; ++i)
            {
                T entry;
                stream >> entry;
                c << entry;
                if (!stream.isValid())
                {
                    c.resize(0);
                    break;
                }
            }
            c.squeeze();
        }
        return stream.isValid();
    }
};

template <>
class SequenceContainer<QStringList>
{
public:
    SequenceContainer() = delete;
    ~SequenceContainer() = delete;
    static bool save(DataStream &stream, const QStringList &c)
    {
        if (!stream.isValid())
            return false;
        stream << static_cast<quint64>(c.size());
        for (const auto &str : c)
            stream << str;
        return stream.isValid();
    }

    static bool load(DataStream &stream, QStringList &c)
    {
        c.clear();
        if (!stream.isValid())
            return false;
        quint64 size;
        if (!HelperMethods::readFromStream(stream, size))
            return false;
        if (0 < size)
        {
            if (!HelperMethods::canReadBytesFromStream(stream, size))
            {
                stream.setValid(false);
                return false;
            }
            c.reserve(qMin<Cute::SizeType>(stream.containerReserveLimit(), size));
            for (quint64 i = 0; i < size; ++i)
            {
                QString str;
                stream >> str;
                c << str;
                if (!stream.isValid())
                {
                    c.clear();
                    break;
                }
            }
        }
        return stream.isValid();
    }
};
#endif

template <typename T>
class SequenceContainer<QList<T>>
{
public:
    SequenceContainer() = delete;
    ~SequenceContainer() = delete;
    static bool save(DataStream &stream, const QList<T> &c)
    {
        if (!stream.isValid())
            return false;
        stream << static_cast<quint64>(c.size());
        QListIterator<T> i(c);
        while (i.hasNext())
            stream << i.next();
        return stream.isValid();
    }

    static bool load(DataStream &stream, QList<T> &c)
    {
#if QT_VERSION < QT_VERSION_CHECK(6,0,0)
        c.clear();
#else
        c.resize(0);
#endif
        if (!stream.isValid())
            return false;
        quint64 size;
        if (!HelperMethods::readFromStream(stream, size))
            return false;
        if (0 < size)
        {
            if (!HelperMethods::canReadBytesFromStream(stream, size))
            {
                stream.setValid(false);
                return false;
            }
            c.reserve(qMin<Cute::SizeType>(stream.containerReserveLimit(), static_cast<Cute::SizeType>(size)));
            for (quint64 i = 0; i < size; ++i)
            {
                T entry;
                stream >> entry;
                c.append(entry);
                if (!stream.isValid())
                {
#if QT_VERSION < QT_VERSION_CHECK(6,0,0)
                    c.clear();
#else
                    c.resize(0);
#endif
                    break;
                }
            }
#if QT_VERSION >= QT_VERSION_CHECK(6,0,0)
            c.squeeze();
#endif
        }
        return stream.isValid();
    }
};

template <typename T>
class SequenceContainer<std::vector<T>>
{
public:
    SequenceContainer() = delete;
    ~SequenceContainer() = delete;
    static bool save(DataStream &stream, const std::vector<T> &c)
    {
        if (!stream.isValid())
            return false;
        stream << static_cast<quint64>(c.size());
        for (auto it = c.cbegin(); it != c.cend(); ++it)
            stream << *it;
        return stream.isValid();
    }

    static bool load(DataStream &stream, std::vector<T> &c)
    {
        c.clear();
        if (!stream.isValid())
            return false;
        quint64 size;
        if (!HelperMethods::readFromStream(stream, size))
            return false;
        if (0 < size)
        {
            if (!HelperMethods::canReadBytesFromStream(stream, size))
            {
                stream.setValid(false);
                return false;
            }
            c.reserve(qMin<Cute::SizeType>(stream.containerReserveLimit(), static_cast<Cute::SizeType>(size)));
            for (quint64 i = 0; i < size; ++i)
            {
                T entry;
                stream >> entry;
                c.push_back(entry);
                if (!stream.isValid())
                {
                    c.resize(0);
                    break;
                }
            }
            c.shrink_to_fit();
        }
        return stream.isValid();
    }
};

template <typename T>
class SequenceContainer<std::list<T>>
{
public:
    SequenceContainer() = delete;
    ~SequenceContainer() = delete;
    static bool save(DataStream &stream, const std::list<T> &c)
    {
        if (!stream.isValid())
            return false;
        stream << static_cast<quint64>(c.size());
        for (auto it = c.cbegin(); it != c.cend(); ++it)
            stream << *it;
        return stream.isValid();
    }

    static bool load(DataStream &stream, std::list<T> &c)
    {
        c.clear();
        if (!stream.isValid())
            return false;
        quint64 size;
        if (!HelperMethods::readFromStream(stream, size))
            return false;
        if (0 < size)
        {
            if (!HelperMethods::canReadBytesFromStream(stream, size))
            {
                stream.setValid(false);
                return false;
            }
            for (quint64 i = 0; i < size; ++i)
            {
                T entry;
                stream >> entry;
                c.push_back(entry);
                if (!stream.isValid())
                {
                    c.clear();
                    break;
                }
            }
        }
        return stream.isValid();
    }
};

template <typename T>
class SequenceContainer<std::deque<T>>
{
public:
    SequenceContainer() = delete;
    ~SequenceContainer() = delete;
    static bool save(DataStream &stream, const std::deque<T> &c)
    {
        if (!stream.isValid())
            return false;
        stream << static_cast<quint64>(c.size());
        for (auto it = c.cbegin(); it != c.cend(); ++it)
            stream << *it;
        return stream.isValid();
    }

    static bool load(DataStream &stream, std::deque<T> &c)
    {
        c.clear();
        if (!stream.isValid())
            return false;
        quint64 size;
        if (!HelperMethods::readFromStream(stream, size))
            return false;
        if (0 < size)
        {
            if (!HelperMethods::canReadBytesFromStream(stream, size))
            {
                stream.setValid(false);
                return false;
            }
            for (quint64 i = 0; i < size; ++i)
            {
                T entry;
                stream >> entry;
                c.push_back(entry);
                if (!stream.isValid())
                {
                    c.clear();
                    break;
                }
            }
            c.shrink_to_fit();
        }
        return stream.isValid();
    }
};

// Unordered set containers (QSet, std::unordered_set)
template <typename T>
class UnorderedSetContainer
{
    static_assert(sizeof(T) == 0, "UnorderedSetContainer only supports specializations for the following types: QSet and std::unordered_set.");
};

template<typename T>
class UnorderedSetContainer<QSet<T>>
{
public:
    UnorderedSetContainer() = delete;
    ~UnorderedSetContainer() = delete;

    static bool save(DataStream &stream, const QSet<T> &s)
    {
        if (!stream.isValid())
            return false;
        stream << static_cast<quint64>(s.size());
        QSetIterator<T> i(s);
        while (i.hasNext())
            stream << i.next();
        return stream.isValid();
    }

    static bool load(DataStream &stream, QSet<T> &s)
    {
        s.clear();
        if (!stream.isValid())
            return false;
        quint64 size;
        if (!HelperMethods::readFromStream(stream, size))
            return false;
        if (0 < size)
        {
            if (!HelperMethods::canReadBytesFromStream(stream, size))
            {
                stream.setValid(false);
                return false;
            }
            s.reserve(qMin<Cute::SizeType>(stream.containerReserveLimit(), static_cast<Cute::SizeType>(size)));
            for (quint64 i = 0; i < size; ++i)
            {
                T entry;
                stream >> entry;
                s.insert(entry);
                if (!stream.isValid())
                {
                    s.clear();
                    break;
                }
            }
            s.squeeze();
        }
        return stream.isValid();
    }
};

template<typename T>
class UnorderedSetContainer<std::unordered_set<T>>
{
public:
    UnorderedSetContainer() = delete;
    ~UnorderedSetContainer() = delete;

    static bool save(DataStream &stream, const std::unordered_set<T> &s)
    {
        if (!stream.isValid())
            return false;
        stream << static_cast<quint64>(s.size());
        for (auto it = s.cbegin(); it != s.cend(); ++it)
            stream << *it;
        return stream.isValid();
    }

    static bool load(DataStream &stream, std::unordered_set<T> &s)
    {
        s.clear();
        if (!stream.isValid())
            return false;
        quint64 size;
        if (!HelperMethods::readFromStream(stream, size))
            return false;
        if (0 < size)
        {
            if (!HelperMethods::canReadBytesFromStream(stream, size))
            {
                stream.setValid(false);
                return false;
            }
            s.reserve(qMin<Cute::SizeType>(stream.containerReserveLimit(), static_cast<Cute::SizeType>(size)));
            for (quint64 i = 0; i < size; ++i)
            {
                T entry;
                stream >> entry;
                s.insert(entry);
                if (!stream.isValid())
                {
                    s.clear();
                    break;
                }
            }
        }
        return stream.isValid();
    }
};


// Stack containers (QStack, std::stack)
template <typename T>
class StackContainer
{
    static_assert(sizeof(T) == 0, "StackContainer only supports specializations for the following types: QStack and std::stack.");
};

template<typename T>
class StackContainer<QStack<T>>
{
public:
    StackContainer() = delete;
    ~StackContainer() = delete;

    static bool save(DataStream &stream, const QStack<T> &s)
    {
        if (!stream.isValid())
            return false;
        auto tempStack = s;
        QVector<T> stackElements(s.size());
        for (int i = stackElements.size() - 1; i >= 0; --i)
            stackElements[i] = tempStack.pop();
        stream << stackElements;
        return stream.isValid();
    }

    static bool load(DataStream &stream, QStack<T> &s)
    {
        s.resize(0);
        if (!stream.isValid())
            return false;
        QVector<T> stackElements;
        stream >> stackElements;
        if (stackElements.isEmpty() || !stream.isValid())
            return stream.isValid();
        s.reserve(stackElements.size());
        for (const auto &e: stackElements)
            s.push(e);
        return stream.isValid();
    }
};

template<typename T>
class StackContainer<std::stack<T>>
{
public:
    StackContainer() = delete;
    ~StackContainer() = delete;

    static bool save(DataStream &stream, const std::stack<T> &s)
    {
        if (!stream.isValid())
            return false;
        auto tempStack = s;
        QVector<T> stackElements(static_cast<qsizetype>(s.size()));
        for (int i = stackElements.size() - 1; i >= 0; --i)
        {
            stackElements[i] = tempStack.top();
            tempStack.pop();
        }
        stream << stackElements;
        return stream.isValid();
    }

    static bool load(DataStream &stream, std::stack<T> &s)
    {
        s = {};
        if (!stream.isValid())
            return false;
        QVector<T> stackElements;
        stream >> stackElements;
        if (stackElements.isEmpty() || !stream.isValid())
            return stream.isValid();
        for (const auto &e: stackElements)
            s.push(e);
        return stream.isValid();
    }
};

// Queue containers (QQueue, std::queue)
template <typename T>
class QueueContainer
{
    static_assert(sizeof(T) == 0, "QueueContainer only supports specializations for the following types: QQueue and std::queue.");
};

template<typename T>
class QueueContainer<QQueue<T>>
{
public:
    QueueContainer() = delete;
    ~QueueContainer() = delete;

    static bool save(DataStream &stream, const QQueue<T> &q)
    {
        if (!stream.isValid())
            return false;
        auto tempQueue = q;
        QVector<T> queueElements(q.size());
        for (int i = 0; i < queueElements.size(); ++i)
            queueElements[i] = tempQueue.dequeue();
        stream << queueElements;
        return stream.isValid();
    }

    static bool load(DataStream &stream, QQueue<T> &s)
    {
        s.clear();
        if (!stream.isValid())
            return false;
        QVector<T> queueElements;
        stream >> queueElements;
        if (queueElements.isEmpty() || !stream.isValid())
            return stream.isValid();
        s.reserve(queueElements.size());
        for (const auto &e: queueElements)
            s.enqueue(e);
        return stream.isValid();
    }
};

template<typename T>
class QueueContainer<std::queue<T>>
{
public:
    QueueContainer() = delete;
    ~QueueContainer() = delete;

    static bool save(DataStream &stream, const std::queue<T> &q)
    {
        if (!stream.isValid())
            return false;
        auto tempQueue = q;
        QVector<T> queueElements(static_cast<qsizetype>(q.size()));
        for (int i = 0; i < queueElements.size(); ++i)
        {
            queueElements[i] = tempQueue.front();
            tempQueue.pop();
        }
        stream << queueElements;
        return stream.isValid();
    }

    static bool load(DataStream &stream, std::queue<T> &q)
    {
        q = {};
        if (!stream.isValid())
            return false;
        QVector<T> queueElements;
        stream >> queueElements;
        if (queueElements.isEmpty() || !stream.isValid())
            return stream.isValid();
        for (const auto &e: queueElements)
            q.push(e);
        return stream.isValid();
    }
};

// Associative containers (QMap, QHash, std::map, std::unordered_map, QMultiMap, QMultiHash, std::multimap, std::unordered_multimap)
template <typename T>
class AssociativeContainer
{
    static_assert(sizeof(T) == 0, "AssociativeContainer only supports specializations for the following types: QMap, QHash, std::map, std::unordered_map, QMultiMap, QMultiHash, std::multimap and std::unordered_multimap.");
};

template<typename Key, typename T>
class AssociativeContainer<QMap<Key, T>>
{
public:
    AssociativeContainer() = delete;
    ~AssociativeContainer() = delete;

    static bool save(DataStream &stream, const QMap<Key, T> &map)
    {
        if (!stream.isValid())
            return false;
        stream << static_cast<quint64>(map.size());
        auto valueEnd = map.cbegin();
        while (valueEnd != map.cend())
        {
            auto valueFirst = valueEnd;
            while (valueEnd != map.cend() && valueEnd.key() == valueFirst.key())
                ++valueEnd;
            for (int i = std::distance(valueFirst, valueEnd) - 1; i >= 0; --i)
            {
                const auto entry = std::next(valueFirst, i);
                stream << entry.key() << entry.value();
            }
        }
        return stream.isValid();
    }

    static bool load(DataStream &stream, QMap<Key, T> &map)
    {
        map.clear();
        if (!stream.isValid())
            return false;
        quint64 size;
        if (!HelperMethods::readFromStream(stream, size))
            return false;
        if (!HelperMethods::canReadBytesFromStream(stream, 2 * map.size()))
        {
            stream.setValid(false);
            return false;
        }
        for (quint64 i = 0; i < size; ++i)
        {
            Key key;
            stream >> key;
            T value;
            stream >> value;
#if QT_VERSION < QT_VERSION_CHECK(6,0,0)
            QT_WARNING_PUSH
            QT_WARNING_DISABLE_DEPRECATED
            map.insertMulti(key, value);
            QT_WARNING_POP
#else
            if (map.contains(key))
            {
                map.clear();
                qCritical("DataStream failed to read QMap. Device has multiple elements with same key. Use QMultiMap instead.");
                stream.setValid(false);
                return false;
            }
            else
                map.insert(key, value);
#endif
            if (!stream.isValid())
            {
                map.clear();
                break;
            }
        }
        return stream.isValid();
    }
};

template<typename Key, typename T>
class AssociativeContainer<QHash<Key, T>>
{
public:
    AssociativeContainer() = delete;
    ~AssociativeContainer() = delete;

    static bool save(DataStream &stream, const QHash<Key, T> &hash)
    {
        if (!stream.isValid())
            return false;
        stream << static_cast<quint64>(hash.size());
        auto valueEnd = hash.cbegin();
        while (valueEnd != hash.cend())
        {
            auto valueFirst = valueEnd;
            while (valueEnd != hash.cend() && valueEnd.key() == valueFirst.key())
                ++valueEnd;
            for (int i = std::distance(valueFirst, valueEnd) - 1; i >= 0; --i)
            {
                const auto entry = std::next(valueFirst, i);
                stream << entry.key() << entry.value();
            }
        }
        return stream.isValid();
    }

    static bool load(DataStream &stream, QHash<Key, T> &hash)
    {
        hash.clear();
        if (!stream.isValid())
            return false;
        quint64 size;
        if (!HelperMethods::readFromStream(stream, size))
            return false;
        if (!HelperMethods::canReadBytesFromStream(stream, 2 * hash.size()))
        {
            stream.setValid(false);
            return false;
        }
        hash.reserve(qMin<Cute::SizeType>(stream.containerReserveLimit(), static_cast<SizeType>(size)));
        for (quint64 i = 0; i < size; ++i)
        {
            Key key;
            stream >> key;
            T value;
            stream >> value;
#if QT_VERSION < QT_VERSION_CHECK(6,0,0)
            QT_WARNING_PUSH
            QT_WARNING_DISABLE_DEPRECATED
            hash.insertMulti(key, value);
            QT_WARNING_POP
#else
            if (hash.contains(key))
            {
                hash.clear();
                qCritical("DataStream failed to read QHash. Device has multiple elements with same key. Use QMultiHash instead.");
                stream.setValid(false);
                return false;
            }
            else
                hash.insert(key, value);
#endif
            if (!stream.isValid())
            {
                hash.clear();
                break;
            }
        }
        return stream.isValid();
    }
};

template<typename Key, typename T>
class AssociativeContainer<std::map<Key, T>>
{
public:
    AssociativeContainer() = delete;
    ~AssociativeContainer() = delete;

    static bool save(DataStream &stream, const std::map<Key, T> &map)
    {
        if (!stream.isValid())
            return false;
        stream << static_cast<quint64>(map.size());
        for (auto it = map.cbegin(); it != map.cend(); ++it)
            stream << it->first << it->second;
        return stream.isValid();
    }

    static bool load(DataStream &stream, std::map<Key, T> &map)
    {
        map.clear();
        if (!stream.isValid())
            return false;
        quint64 size;
        if (!HelperMethods::readFromStream(stream, size))
            return false;
        if (!HelperMethods::canReadBytesFromStream(stream, 2 * map.size()))
        {
            stream.setValid(false);
            return false;
        }
        for (quint64 i = 0; i < size; ++i)
        {
            Key key;
            stream >> key;
            T value;
            stream >> value;
            if (map.count(key))
            {
                map.clear();
                qCritical("DataStream failed to read std::map. Device has multiple elements with same key. Use std::multimap instead.");
                stream.setValid(false);
                return false;
            }
            else
                map.insert(std::make_pair(key, value));
            if (!stream.isValid())
            {
                map.clear();
                break;
            }
        }
        return stream.isValid();
    }
};

template<typename Key, typename T>
class AssociativeContainer<std::unordered_map<Key, T>>
{
public:
    AssociativeContainer() = delete;
    ~AssociativeContainer() = delete;

    static bool save(DataStream &stream, const std::unordered_map<Key, T> &map)
    {
        if (!stream.isValid())
            return false;
        stream << static_cast<quint64>(map.size());
        for (auto it = map.cbegin(); it != map.cend(); ++it)
            stream << it->first << it->second;
        return stream.isValid();
    }

    static bool load(DataStream &stream, std::unordered_map<Key, T> &map)
    {
        map.clear();
        if (!stream.isValid())
            return false;
        quint64 size;
        if (!HelperMethods::readFromStream(stream, size))
            return false;
        if (!HelperMethods::canReadBytesFromStream(stream, 2 * map.size()))
        {
            stream.setValid(false);
            return false;
        }
        map.reserve(qMin<Cute::SizeType>(stream.containerReserveLimit(), static_cast<SizeType>(size)));
        for (quint64 i = 0; i < size; ++i)
        {
            Key key;
            stream >> key;
            T value;
            stream >> value;
            if (map.count(key))
            {
                map.clear();
                qCritical("DataStream failed to read std::unordered_map. Device has multiple elements with same key. Use std::unordered_multimap instead.");
                stream.setValid(false);
                return false;
            }
            else
                map.insert(std::make_pair(key, value));
            if (!stream.isValid())
            {
                map.clear();
                break;
            }
        }
        return stream.isValid();
    }
};

template<typename Key, typename T>
class AssociativeContainer<QMultiMap<Key, T>>
{
public:
    AssociativeContainer() = delete;
    ~AssociativeContainer() = delete;

    static bool save(DataStream &stream, const QMultiMap<Key, T> &map)
    {
        if (!stream.isValid())
            return false;
        stream << static_cast<quint64>(map.size());
        auto valueEnd = map.cbegin();
        while (valueEnd != map.cend())
        {
            auto valueFirst = valueEnd;
            while (valueEnd != map.cend() && valueEnd.key() == valueFirst.key())
                ++valueEnd;
            for (int i = std::distance(valueFirst, valueEnd) - 1; i >= 0; --i)
            {
                const auto entry = std::next(valueFirst, i);
                stream << entry.key() << entry.value();
            }
        }
        return stream.isValid();
    }

    static bool load(DataStream &stream, QMultiMap<Key, T> &map)
    {
        map.clear();
        if (!stream.isValid())
            return false;
        quint64 size;
        if (!HelperMethods::readFromStream(stream, size))
            return false;
        if (!HelperMethods::canReadBytesFromStream(stream, 2 * map.size()))
        {
            stream.setValid(false);
            return false;
        }
        for (quint64 i = 0; i < size; ++i)
        {
            Key key;
            stream >> key;
            T value;
            stream >> value;
            map.insert(key, value);
            if (!stream.isValid())
            {
                map.clear();
                break;
            }
        }
        return stream.isValid();
    }
};

template<typename Key, typename T>
class AssociativeContainer<QMultiHash<Key, T>>
{
public:
    AssociativeContainer() = delete;
    ~AssociativeContainer() = delete;

    static bool save(DataStream &stream, const QMultiHash<Key, T> &hash)
    {
        if (!stream.isValid())
            return false;
        stream << static_cast<quint64>(hash.size());
        auto valueEnd = hash.cbegin();
        while (valueEnd != hash.cend())
        {
            auto valueFirst = valueEnd;
            while (valueEnd != hash.cend() && valueEnd.key() == valueFirst.key())
                ++valueEnd;
            for (int i = std::distance(valueFirst, valueEnd) - 1; i >= 0; --i)
            {
                const auto entry = std::next(valueFirst, i);
                stream << entry.key() << entry.value();
            }
        }
        return stream.isValid();
    }

    static bool load(DataStream &stream, QMultiHash<Key, T> &hash)
    {
        hash.clear();
        if (!stream.isValid())
            return false;
        quint64 size;
        if (!HelperMethods::readFromStream(stream, size))
            return false;
        if (!HelperMethods::canReadBytesFromStream(stream, 2 * hash.size()))
        {
            stream.setValid(false);
            return false;
        }
        hash.reserve(qMin<Cute::SizeType>(stream.containerReserveLimit(), static_cast<SizeType>(size)));
        for (quint64 i = 0; i < size; ++i)
        {
            Key key;
            stream >> key;
            T value;
            stream >> value;
            hash.insert(key, value);
            if (!stream.isValid())
            {
                hash.clear();
                break;
            }
        }
        return stream.isValid();
    }
};

template<typename Key, typename T>
class AssociativeContainer<std::multimap<Key, T>>
{
public:
    AssociativeContainer() = delete;
    ~AssociativeContainer() = delete;

    static bool save(DataStream &stream, const std::multimap<Key, T> &map)
    {
        if (!stream.isValid())
            return false;
        stream << static_cast<quint64>(map.size());
        auto valueEnd = map.cbegin();
        while (valueEnd != map.cend())
        {
            auto valueFirst = valueEnd;
            while (valueEnd != map.cend() && valueEnd->first == valueFirst->first)
                ++valueEnd;
            for (int i = std::distance(valueFirst, valueEnd) - 1; i >= 0; --i)
            {
                const auto entry = std::next(valueFirst, i);
                stream << entry->first << entry->second;
            }
        }
        return stream.isValid();
    }

    static bool load(DataStream &stream, std::multimap<Key, T> &map)
    {
        map.clear();
        if (!stream.isValid())
            return false;
        quint64 size;
        if (!HelperMethods::readFromStream(stream, size))
            return false;
        if (!HelperMethods::canReadBytesFromStream(stream, 2 * map.size()))
        {
            stream.setValid(false);
            return false;
        }
        for (quint64 i = 0; i < size; ++i)
        {
            Key key;
            stream >> key;
            T value;
            stream >> value;
            map.insert(std::make_pair(key, value));
            if (!stream.isValid())
            {
                map.clear();
                break;
            }
        }
        return stream.isValid();
    }
};

template<typename Key, typename T>
class AssociativeContainer<std::unordered_multimap<Key, T>>
{
public:
    AssociativeContainer() = delete;
    ~AssociativeContainer() = delete;

    static bool save(DataStream &stream, const std::unordered_multimap<Key, T> &map)
    {
        if (!stream.isValid())
            return false;
        stream << static_cast<quint64>(map.size());
        auto valueEnd = map.cbegin();
        while (valueEnd != map.cend())
        {
            auto valueFirst = valueEnd;
            while (valueEnd != map.cend() && valueEnd->first == valueFirst->first)
                ++valueEnd;
            for (int i = std::distance(valueFirst, valueEnd) - 1; i >= 0; --i)
            {
                const auto entry = std::next(valueFirst, i);
                stream << entry->first << entry->second;
            }
        }
        return stream.isValid();
    }

    static bool load(DataStream &stream, std::unordered_multimap<Key, T> &map)
    {
        map.clear();
        if (!stream.isValid())
            return false;
        quint64 size;
        if (!HelperMethods::readFromStream(stream, size))
            return false;
        if (!HelperMethods::canReadBytesFromStream(stream, 2 * map.size()))
        {
            stream.setValid(false);
            return false;
        }
        map.reserve(qMin<Cute::SizeType>(stream.containerReserveLimit(), static_cast<SizeType>(size)));
        for (quint64 i = 0; i < size; ++i)
        {
            Key key;
            stream >> key;
            T value;
            stream >> value;
            map.insert(std::make_pair(key, value));
            if (!stream.isValid())
            {
                map.clear();
                break;
            }
        }
        return stream.isValid();
    }
};

template<typename Enum>
DataStream &DataStream::operator<<(QFlags<Enum> e)
{
    if (!isValid())
        return *this;
    return *this << static_cast<qint32>(typename QFlags<Enum>::Int(e));
}

template<typename Enum>
DataStream &DataStream::operator>>(QFlags<Enum> &e)
{
    if (!isValid())
        return *this;
    qint32 i;
    *this >> i;
    e = QFlags<Enum>(QFlag(i));
    return *this;
}

#if QT_VERSION < QT_VERSION_CHECK(6,0,0)
template<typename T>
DataStream &DataStream::operator<<(const QVector<T> &c)
{
    if (!SequenceContainer<QVector<T>>::save(*this, c))
        this->setValid(false);
    return *this;
}

template<typename T>
DataStream &DataStream::operator>>(QVector<T> &c)
{
    if (!SequenceContainer<QVector<T>>::load(*this, c))
        this->setValid(false);
    return *this;
}
#endif

template<typename T>
DataStream &DataStream::operator<<(const QList<T> &list)
{
    if (!SequenceContainer<QList<T>>::save(*this, list))
        this->setValid(false);
    return *this;
}

template<typename T>
DataStream &DataStream::operator>>(QList<T> &list)
{
    if (!SequenceContainer<QList<T>>::load(*this, list))
        this->setValid(false);
    return *this;
}

template <typename T>
DataStream &DataStream::operator<<(const std::vector<T> &v)
{
    if (!SequenceContainer<std::vector<T>>::save(*this, v))
        this->setValid(false);
    return *this;
}

template <typename T>
DataStream &DataStream::operator>>(std::vector<T> &v)
{
    if (!SequenceContainer<std::vector<T>>::load(*this, v))
        this->setValid(false);
    return *this;
}

template <typename T>
DataStream &DataStream::operator<<(const std::list<T> &l)
{
    if (!SequenceContainer<std::list<T>>::save(*this, l))
        this->setValid(false);
    return *this;
}

template <typename T>
DataStream &DataStream::operator>>(std::list<T> &l)
{
    if (!SequenceContainer<std::list<T>>::load(*this, l))
        this->setValid(false);
    return *this;
}

template <typename T>
DataStream &DataStream::operator<<(const std::deque<T> &d)
{
    if (!SequenceContainer<std::deque<T>>::save(*this, d))
        this->setValid(false);
    return *this;
}

template <typename T>
DataStream &DataStream::operator>>(std::deque<T> &d)
{
    if (!SequenceContainer<std::deque<T>>::load(*this, d))
        this->setValid(false);
    return *this;
}

template<typename T>
DataStream &DataStream::operator<<(const QSet<T> &set)
{
    if (!UnorderedSetContainer<QSet<T>>::save(*this, set))
        setValid(false);
    return *this;
}

template<typename T>
DataStream &DataStream::operator>>(QSet<T> &set)
{
    if (!UnorderedSetContainer<QSet<T>>::load(*this, set))
        setValid(false);
    return *this;
}

template<typename T>
DataStream &DataStream::operator<<(const std::unordered_set<T> &set)
{
    if (!UnorderedSetContainer<std::unordered_set<T>>::save(*this, set))
        setValid(false);
    return *this;
}

template<typename T>
DataStream &DataStream::operator>>(std::unordered_set<T> &set)
{
    if (!UnorderedSetContainer<std::unordered_set<T>>::load(*this, set))
        setValid(false);
    return *this;
}

template <class T>
DataStream &DataStream::operator<<(const QStack<T> &s)
{
    if (!StackContainer<QStack<T>>::save(*this, s))
        setValid(false);
    return *this;
}

template <class T>
DataStream &DataStream::operator>>(QStack<T> &s)
{
    if (!StackContainer<QStack<T>>::load(*this, s))
        setValid(false);
    return *this;
}

template <class T>
DataStream &DataStream::operator<<(const std::stack<T> &s)
{
    if (!StackContainer<std::stack<T>>::save(*this, s))
        setValid(false);
    return *this;
}

template <class T>
DataStream &DataStream::operator>>(std::stack<T> &s)
{
    if (!StackContainer<std::stack<T>>::load(*this, s))
        setValid(false);
    return *this;
}

template <typename T>
DataStream &DataStream::operator<<(const QQueue<T> &q)
{
    if (!QueueContainer<QQueue<T>>::save(*this, q))
        setValid(false);
    return *this;
}

template <typename T>
DataStream &DataStream::operator>>(QQueue<T> &q)
{
    if (!QueueContainer<QQueue<T>>::load(*this, q))
        setValid(false);
    return *this;
}

template <typename T>
DataStream &DataStream::operator<<(const std::queue<T> &q)
{
    if (!QueueContainer<std::queue<T>>::save(*this, q))
        setValid(false);
    return *this;
}

template <typename T>
DataStream &DataStream::operator>>(std::queue<T> &q)
{
    if (!QueueContainer<std::queue<T>>::load(*this, q))
        setValid(false);
    return *this;
}

template<class Key, class T>
DataStream &DataStream::operator<<(const QMap<Key, T> &map)
{
    if (!AssociativeContainer<QMap<Key, T>>::save(*this, map))
        setValid(false);
    return *this;
}

template<class Key, class T>
DataStream &DataStream::operator>>(QMap<Key, T> &map)
{
    if (!AssociativeContainer<QMap<Key, T>>::load(*this, map))
        setValid(false);
    return *this;
}

template<class Key, class T>
DataStream &DataStream::operator<<(const QHash<Key, T> &hash)
{
    if (!AssociativeContainer<QHash<Key, T>>::save(*this, hash))
        setValid(false);
    return *this;
}

template<class Key, class T>
DataStream &DataStream::operator>>(QHash<Key, T> &hash)
{
    if (!AssociativeContainer<QHash<Key, T>>::load(*this, hash))
        setValid(false);
    return *this;
}

template<class Key, class T>
DataStream &DataStream::operator<<(const std::map<Key, T> &map)
{
    if (!AssociativeContainer<std::map<Key, T>>::save(*this, map))
        setValid(false);
    return *this;
}

template<class Key, class T>
DataStream &DataStream::operator>>(std::map<Key, T> &map)
{
    if (!AssociativeContainer<std::map<Key, T>>::load(*this, map))
        setValid(false);
    return *this;
}

template<class Key, class T>
DataStream &DataStream::operator<<(const std::unordered_map<Key, T> &map)
{
    if (!AssociativeContainer<std::unordered_map<Key, T>>::save(*this, map))
        setValid(false);
    return *this;
}

template<class Key, class T>
DataStream &DataStream::operator>>(std::unordered_map<Key, T> &map)
{
    if (!AssociativeContainer<std::unordered_map<Key, T>>::load(*this, map))
        setValid(false);
    return *this;
}

template<class Key, class T>
DataStream &DataStream::operator<<(const QMultiMap<Key, T> &map)
{
    if (!AssociativeContainer<QMultiMap<Key, T>>::save(*this, map))
        setValid(false);
    return *this;
}

template<class Key, class T>
DataStream &DataStream::operator>>(QMultiMap<Key, T> &map)
{
    if (!AssociativeContainer<QMultiMap<Key, T>>::load(*this, map))
        setValid(false);
    return *this;
}

template<class Key, class T>
DataStream &DataStream::operator<<(const QMultiHash<Key, T> &hash)
{
    if (!AssociativeContainer<QMultiHash<Key, T>>::save(*this, hash))
        setValid(false);
    return *this;
}

template<class Key, class T>
DataStream &DataStream::operator>>(QMultiHash<Key, T> &hash)
{
    if (!AssociativeContainer<QMultiHash<Key, T>>::load(*this, hash))
        setValid(false);
    return *this;
}

template <class Key, class T>
DataStream &DataStream::operator<<(const std::multimap<Key, T> &map)
{
    if (!AssociativeContainer<std::multimap<Key, T>>::save(*this, map))
        setValid(false);
    return *this;
}

template <class Key, class T>
DataStream &DataStream::operator>>(std::multimap<Key, T> &map)
{
    if (!AssociativeContainer<std::multimap<Key, T>>::load(*this, map))
        setValid(false);
    return *this;
}

template <class Key, class T>
DataStream &DataStream::operator<<(const std::unordered_multimap<Key, T> &map)
{
    if (!AssociativeContainer<std::unordered_multimap<Key, T>>::save(*this, map))
        setValid(false);
    return *this;
}

template <class Key, class T>
DataStream &DataStream::operator>>(std::unordered_multimap<Key, T> &map)
{
    if (!AssociativeContainer<std::unordered_multimap<Key, T>>::load(*this, map))
        setValid(false);
    return *this;
}

template<typename T1, typename T2>
DataStream &DataStream::operator<<(const std::pair<T1, T2> &p)
{
    if (!isValid())
        return *this;
    return *this << p.first << p.second;
}

template<typename T1, typename T2>
DataStream &DataStream::operator>>(std::pair<T1, T2> &p)
{
    if (!isValid())
        return *this;
    return *this >> p.first >> p.second;
}

}

#endif // CUTE_DATA_STREAM_H
