//
// Copyright (c) 2020 Glauco Pacheco <glauco@cuteserver.io>
// All rights reserved
//

#ifndef CUTE_ERROR_H
#define CUTE_ERROR_H

#include "ConnectionInfo.h"
#include "DataStream/DataStream.h"
#include <QObject>
#include <QSslSocket>

namespace Cute
{

enum class ErrorType {Socket, SSL, Connection};

enum class ConnectionError
{
    InvalidProxyHeader,
    ProxyHeaderReadTimedOut,
    SslHandshakeTimedOut,
    WebSocketHandshakeTimedOut,
    CuteHandshakeTimedOut,
    CuteHandshakeFailed,
    RequestTimedOut,
    ConnectionTimedOut,
    MessageIntegrityCheckFailed,
    InvalidHttpRequest,
    HttpHandlerNotFound,
    InvalidMessage,
    RemoteObjectNotFound,
    RemoteSignalNotFound,
    RemoteSlotNotFound,
    TooBigSignal,
    InvalidObject,
    InvalidSignal,
    InvalidSlot,
    IncompatibleSignalSlot,
    InvalidSignalEmitter,
    RemoteSlotRaisedException,
    TooManyRemoteObjectsPerConnection,
    TooManyRemoteSignalSlotConnections,
    InvalidRemoteObjectUrl,
    ConnectionDropped,
    ConnectedReceiverNotFound,
    InvalidArgType,
    NotRegisteredArgument,
    UnresponsivePeer
};

union Error
{
    QAbstractSocket::SocketError socketError;
    QSslError::SslError sslError;
    ConnectionError connectionError;
};

struct PUBLIC ErrorInfo
{
    ErrorInfo();
    ErrorInfo(ErrorType errorType, Error errorError);
    ~ErrorInfo() = default;
    ErrorType type = ErrorType::Socket;
    Error error = {.socketError = QAbstractSocket::UnknownSocketError};
    [[nodiscard]] QString description() const;
};

PUBLIC bool operator==(const ErrorInfo &ei1, const ErrorInfo &ei2);
PUBLIC bool operator!=(const ErrorInfo &ei1, const ErrorInfo &ei2);
PUBLIC DataStream &operator<<(DataStream &stream, const ErrorInfo &errorInfo);
PUBLIC DataStream &operator>>(DataStream &stream, ErrorInfo &errorInfo);
PUBLIC ErrorInfo makeErrorInfo(QAbstractSocket::SocketError error);
PUBLIC ErrorInfo makeErrorInfo(QSslError::SslError error);
PUBLIC ErrorInfo makeErrorInfo(ConnectionError error);
PUBLIC QString errorInfoDescription(ErrorInfo errorInfo);

class IErrorHandler
{
public:
    IErrorHandler() = default;
    virtual ~IErrorHandler() = default;
    virtual void handleError(const ErrorInfo &errorInfo, const ConnectionInfo &connectionInfo) = 0;
};

PUBLIC void setErrorHandler(IErrorHandler *errorHandlerInstance);

}

Q_DECLARE_METATYPE(Cute::ErrorInfo)

#endif // CUTE_ERROR_H
